<?php
	
	/*
		
		==================================================================
		
			KNET PHP SDK
			
			Developed on the 1st of July, 2019
			
			Developer: Abdullah S. Al-Salloum
			E-mail: contact@abdullah.com.kw
			Website: https://knet-php-sdk.abdullah.com.kw
			
			Kuwait		

		==================================================================
			
		- TRANSPORTAL_ID: 
		
		Your KNET terminal transportal ID. Go to KNET control panel, then look for information about your terminal.
		
		e.g.: "000000"

		---------------------------------------
			
		- TRANSPORTAL_PASSWORD: 
		
		Your KNET terminal transportal password. Go to KNET control panel, download your terminal resource.cgn and keystore.bin.
		Then email both files to pgsupport@knet.com.kw asking them to extract your terminal transportal password from resource.cgn.
		
		e.g.: "000000"

		---------------------------------------

		- TERMINAL_RESOURCE_KEY: 
		
		Your KNET terminal transportal resource key. Go to KNET control panel, download your terminal resource.cgn and keystore.bin.
		Then email both files to pgsupport@knet.com.kw asking them to extract your terminal resource key from keystore.bin.
		
		e.g.: "000000"

		---------------------------------------

		- TERMINAL_URI: 
		
		If you are in the test phase, use: "https://kpaytest.com.kw".
		For production (live), use: "https://kpay.com.kw".
		
		e.g.: "https://kpaytest.com.kw"

		---------------------------------------
									
		-CURRENCY: 
		
		The currency of the transaction.
		
		e.g.: "414" means KWD
		
		---------------------------------------
	
		-DECIMALS: 
		
		The currency unit decimals, "3" for KWD.
		
		e.g.: "3" means 0.000

		---------------------------------------
		
		- LISTEN_URI: 
		
		This is where you must use listen() followed
		by validation followed by respond(). This
		must be accessible over HTTPS with a valid
		SSL certificate.
		
		e.g.: "https://domain.com/path/to/callback/page"


		---------------------------------------
	
		- COMPLETE_URI: 
		
		This is the final destination where the payee
		will be redirected to. This should display the
		status of the payment based on the TRACKING_ID
		given in GET. HTTPS is not mandatory for this
		file.
		
		e.g.: "https://domain.com/path/to/completion/page"
		
		---------------------------------------

		- ACTION: 
		
		MUST be "1" as purchase.
		
		---------------------------------------

		- DEBUG: 
		
		Either TRUE or FALSE. If set to TRUE then
		you will be using a virtual gateway to test
		your code. Set this to false if you prefer
		testing with KNET directly.

		==================================================================

		$k = new KNET([PARAMETERS]);
		
		Defines the class.
		
		[PARAMETERS]:
			array(
				"TRANSPORTAL_ID" => "000000", // Transportal ID
				"TRANSPORTAL_PASSWORD" => "000000", // Transportal password
				"TERMINAL_RESOURCE_KEY" => "000000", // Terminal resource key
				"TERMINAL_URI" => "https://kpaytest.com.kw", // Terminal URI, "https://kpaytest.com.kw" for testing, "https://kpay.com.kw" for production
				"CURRENCY" => "414",
				"DECIMALS" => "3",
				"LISTEN_URI" => "https://LISTEN_URI.page",
				"COMPLETE_URI" => "https://COMPLETE_URI.page",
				"ACTION" => "1",
				"DEBUG" => FALSE
			)
			
		[RETURNS]:
			VOID

		==================================================================

		$k->init();
		
		Defines the class.
		
		[PARAMETERS]:
			VOID
			
		[RETURNS]:
			- On success:
				array(
					"STATUS" => "success",
					"DESCRIPTION" => "DESCRIPTIVE_MESSAGE"
				)
			- On error:
				array(
					"STATUS" => "error",
					"DESCRIPTION" => "DESCRIPTIVE_MESSAGE"
				)

		==================================================================
		
		$k->valid();
		
		Tells whether the initialization is valid.
		
		[PARAMETERS]:
			VOID
			
		[RETURNS]:
			- On success:
				bool(true)
			- On error:
				bool(false)

		==================================================================
		
		$k->create([PARAMETERS]);
		
		Creates a payment.
		
		[PARAMETERS]:
			array(
				"AMOUNT" => "10.000",
				"LANGUAGE" => "ENG",
				"USER_DEFINED_VAR_1" => "Any variable to be received in listen()",
				"USER_DEFINED_VAR_2" => "Any variable to be received in listen()",
				"USER_DEFINED_VAR_3" => "Any variable to be received in listen()",
				"USER_DEFINED_VAR_4" => "Any variable to be received in listen()"
			)
		
		[RETURNS]:
			- On success:
				array(
					"STATUS" => "success",
					"TRACKING_ID" => "UNIQUE_TRACKING_ID",
					"PAYMENT_URI" => "GATEWAY_PAYMENT_URI"
				)
			- On error:
				array(
					"STATUS" => "error",
					"DESCRIPTION" => "ERROR_DESCRIPTION"
				)
		
		==================================================================
		
		$k->start();
		
		Sends the payee to the payment gateway. This can only be
		called after create();
		
		[PARAMETERS]:
			VOID
		
		[RETURNS]:
			- On success:
				bool(true)
			- On error:
				bool(false)
		
		==================================================================
		
		$k->listen();
		
		This is to be placed at the top of LISTEN_URI.
		
		[PARAMETERS]:
			All POST variables coming from the gateway.
		
		[RETURNS]:
			- On success:
				array(
					"STATUS" => "success",
					"PAYMENT_STATUS" => "either 'successful' or 'failed'",
					"DESCRIPTION" => "Descriptive status",
					"TRACKING_ID" => "The payment tracking identification number",
					"PAYMENT_ID" => "The payment identification number",
					"AMOUNT" => "The payment amount",
					"TRANSACTION_ID" => "Gateway transaction identification number",
					"REFERENCE_ID" => "Gateway reference identification number",
					"AUTHORIZATION_ID" => "Gateway authorization identification number",
					"USER_DEFINED_VAR_1" => "Variable passed to create()",
					"USER_DEFINED_VAR_2" => "Variable passed to create()",
					"USER_DEFINED_VAR_3" => "Variable passed to create()",
					"USER_DEFINED_VAR_4" => "Variable passed to create()",
					"PAYMENT_TIME" => "Payment date & time"
				)
			- On error:
				array(
					"STATUS" => "error",
					"DESCRIPTION" => "ERROR_DESCRIPTION"
				)
			
		==================================================================
		
		$k->respond();
		
		This is to be placed at the end of LISTEN_URI after
		validating local records with payment data given by
		listen();			
			
		[RETURNS]:
			- On success:
				REDIRECT TO "[COMPLETE_URI]?TRACKING_ID=[TRACKING_ID]"
			- On error:
				REDIRECT TO "[COMPLETE_URI]?TRACKING_ID=[TRACKING_ID]"
		
		==================================================================
	
	*/
	
	class Knet {

		private $DEBUG = FALSE;
		private $VIRTUAL_GATEWAY = "https://knet-php-sdk.abdullah.com.kw/virtualgateway"; // URL of virtual gateway
		private $TRANSPORTAL_ID;
		private $TRANSPORTAL_PASSWORD;
		private $TERMINAL_RESOURCE_KEY;
		private $TERMINAL_URI;
		private $CURRENCY;
		private $DECIMALS;
		private $LISTEN_URI;
		private $COMPLETE_URI;
		private $ACTION;		
		private $RESPONSETOKEN;
		private $passedAccessValidation;
		private $TRACKING_ID;
		private $PAYMENT_URI;
		private $SUCCESSFULLISTEN = false;
		private $TRANSACTION_RESULT;

		public function __construct($p) {
			$this->TRANSPORTAL_ID = $p["TRANSPORTAL_ID"];
			$this->TRANSPORTAL_PASSWORD = $p["TRANSPORTAL_PASSWORD"];
			$this->TERMINAL_RESOURCE_KEY = $p["TERMINAL_RESOURCE_KEY"];
			$this->TERMINAL_URI = $p["TERMINAL_URI"];
			$this->CURRENCY = $p["CURRENCY"];
			$this->DECIMALS = $p["DECIMALS"];
			$this->LISTEN_URI = $p["LISTEN_URI"];
			$this->COMPLETE_URI = $p["COMPLETE_URI"];
			$this->ACTION = $p["ACTION"];
			$this->DEBUG = ($p["DEBUG"] == TRUE) ? TRUE : FALSE;
		}

		/* =================================================================================== Public Methods */
		
		public function init() {
			$r = $this->validateAccessData();
			$this->passedAccessValidation = $r["STATUS"];
			return $r;		
		}
		
		public function valid() {
			if($this->passedAccessValidation == "success") return true;
			else return false;
		}

		public function start() {
			if($this->isValid($this->PAYMENT_URI,"URI")) {
				header("Location: ".$this->PAYMENT_URI);
				return true;
			} else {
				return false;
			}
		}
		
		public function create($p) {
			if(!$this->valid()) {
				$r["STATUS"] = "error";
				$r["DESCRIPTION"] = "Validation of access data failed. Check the output of init().";			
			} else {
				$p = $this->validateInitializationData($p);
				if($p["STATUS"] != "passed") {
					$r["STATUS"] = "error";
					$r["DESCRIPTION"] = $p["DESCRIPTION"];			
				} else {
					$this->cTrackingID($p);
					$d = [
						"amt" => $p["AMOUNT"],
						"trackid" => $this->TRACKING_ID,
						"id" => $this->TRANSPORTAL_ID,
						"password" => $this->TRANSPORTAL_PASSWORD,
						"action" => $this->ACTION,
						"currencycode" => $this->CURRENCY,
						"langid" => $p["LANGUAGE"],
						"responseURL" => $this->LISTEN_URI,
						"errorURL" => $this->LISTEN_URI,
						"udf1" => $p["USER_DEFINED_VAR_1"],
						"udf2" => $p["USER_DEFINED_VAR_2"],
						"udf3" => $p["USER_DEFINED_VAR_3"],
						"udf4" => $p["USER_DEFINED_VAR_4"],
						"udf5" => $this->RESPONSETOKEN
					];
					
					$rp = "&trandata=".$this->textEncrypt($this->array2Args($d),$this->TERMINAL_RESOURCE_KEY)."&tranportalId=".$this->TRANSPORTAL_ID."&responseURL=".$this->LISTEN_URI."&errorURL=".$this->LISTEN_URI;					
					if($this->DEBUG == TRUE) $ruri = $this->VIRTUAL_GATEWAY."/?param=paymentInit&trk=".$this->TERMINAL_RESOURCE_KEY.$rp;
					else $ruri = $this->TERMINAL_URI . "/kpg/PaymentHTTP.htm?param=paymentInit".$rp;
					
					$r["STATUS"] = "success";
					$r["TRACKING_ID"] = $this->TRACKING_ID;
					$r["PAYMENT_URI"] = $this->PAYMENT_URI = $ruri;
				}
			}
			return $r;
		}
		
		public function listen() {
			$vars = ($this->DEBUG == TRUE) ? $_REQUEST : $_POST;
			if(!$this->valid()) {
				$this->TRANSACTION_RESULT = "internal_failure";
				$r["STATUS"] = "error";
				$r["DESCRIPTION"] = "Validation of initialization data failed. Check the output of init().";			
			} elseif(!isset($vars["ErrorText"]) && !isset($vars["Error"])) {
				
				$this->TRACKING_ID = (trim($vars["trackid"]) != "") ? trim($vars["trackid"]) : trim($_GET["trackid"]);
				
				parse_str($this->textDecrypt($vars["trandata"],$this->TERMINAL_RESOURCE_KEY),$vars);
				$this->TRANSACTION_RESULT = (trim($vars["result"]) != "") ? trim($vars["result"]) : trim($vars["Result"]);
				if($this->TRANSACTION_RESULT == "CANCELED") {
					$this->SUCCESSFULLISTEN = true;
					$r["STATUS"] = "success";
					$r["PAYMENT_STATUS"] = "failed";
					$r["TRACKING_ID"] = $this->TRACKING_ID;
					$r["DESCRIPTION"] = "Payment failed processing from the gateway side. Gateway response is: [[[".$this->TRANSACTION_RESULT."]]]";		
				} else {
					if(trim($vars["udf5"]) != $this->RESPONSETOKEN) {
						$this->TRANSACTION_RESULT = "gateway_failure";
						$r["STATUS"] = "success";
						$r["PAYMENT_STATUS"] = "failed";
						$r["TRACKING_ID"] = $this->TRACKING_ID;
						$r["DESCRIPTION"] = "Unauthorized access or the gateway had technical issues processing this payment.";							
					} else {

						$r["STATUS"] = "success";
						$this->SUCCESSFULLISTEN = true;
						if($this->TRANSACTION_RESULT == "CAPTURED") {
							$r["PAYMENT_STATUS"] = "successful";
							$r["DESCRIPTION"] ="Payment has been processed successfully. Gateway response is: [[[".$this->TRANSACTION_RESULT."]]]";
						} else {
							$r["PAYMENT_STATUS"] = "failed";
							$r["DESCRIPTION"] = "Payment failed processing from the gateway side. Gateway response is: [[[".$this->TRANSACTION_RESULT."]]]";
						}
						$r["TRACKING_ID"] = trim($vars["trackid"]);
						$r["PAYMENT_ID"] = trim($vars["paymentid"]);
						$r["TRANSACTION_ID"] = trim($vars["tranid"]);	
						$r["AMOUNT"] = trim($vars["amt"]);	
						$r["REFERENCE_ID"] = trim($vars["ref"]);	
						$r["AUTHORIZATION_ID"] = trim($vars["auth"]);	
						$r["USER_DEFINED_VAR_1"] = trim($vars["udf1"]);	
						$r["USER_DEFINED_VAR_2"] = trim($vars["udf2"]);	
						$r["USER_DEFINED_VAR_3"] = trim($vars["udf3"]);	
						$r["USER_DEFINED_VAR_4"] = trim($vars["udf4"]);	
						$r["PAYMENT_TIME"] = (isset($vars["PostDate"])) ? trim($vars["PostDate"]) : trim($vars["postdate"]);
						if($r["PAYMENT_TIME"] == "") trim($vars["postDate"]);
													
					}
				}
			} else {
				$this->SUCCESSFULLISTEN = true;
				$this->TRANSACTION_RESULT = "gateway_failure";
				$r["STATUS"] = "success";
				$r["PAYMENT_STATUS"] = "failed";
				$r["TRACKING_ID"] = $this->TRACKING_ID = $vars["trackid"];
				$r["DESCRIPTION"] = $vars["Error"] . ": " .$vars["ErrorText"];
			}		

			return $r;		
		}
		
		public function respond(){
			$u = (strpos($this->COMPLETE_URI,"?") !== false) ? $this->COMPLETE_URI . "&" : $this->COMPLETE_URI . "?";
			$u .= "TRACKING_ID=".$this->TRACKING_ID;
			header("Location: ".$u);
		}

		/* =================================================================================== Private Methods */
		
		private function cTrackingID($p) {
			$this->TRACKING_ID = time()."_".md5(time().rand(10000, 999999).$p["AMOUNT"].$p["LANGUAGE"].$p["USER_DEFINED_VAR_2"].$p["USER_DEFINED_VAR_2"].$p["USER_DEFINED_VAR_3"].$p["USER_DEFINED_VAR_4"]);
		}
		
		private function isValid($s,$t) {
			if($t == "USER_DEFINED_VAR" && (preg_match('/^[a-zA-Z0-9_-]{1,}$/', $s) || $s == "")) return true;
			elseif($t == "TRANSPORTAL_ID" && is_numeric($s) && $s > 0) return true;
			elseif(($t == "TRANSPORTAL_PASSWORD" || $t == "TERMINAL_RESOURCE_KEY" || $t == "TERMINAL_RESOURCE_KEY") && $s != "") return true;
			elseif($t == "AMOUNT" && is_numeric($s) && $s > 0) return true;
			elseif($t == "CURRENCYCODE" && is_numeric($s) && $s > 0 && strlen($s) == 3 && strpos($s,".") === false) return true;
			elseif($t == "DECIMALSCODE" && ($s == "0" || $s == "1" || $s == "2" || $s == "3")) return true;
			elseif($t == "ACTION" && ($s == "0" || $s == "1")) return true;
			elseif($t == "LANGUAGE" && ($s == "ENG" || $s == "ARA")) return true;
			elseif($t == "URI" && filter_var($s, FILTER_VALIDATE_URL)) return true;
			else return false;
		}
			
		private function validateAccessData() {
			if(!$this->isValid($this->TRANSPORTAL_ID,"TRANSPORTAL_ID") && $this->DEBUG != TRUE) {
				$r["STATUS"] = "error";
				$r["DESCRIPTION"] = "The transportal id is invalid. Given is: ".$this->TRANSPORTAL_ID;
			} elseif(!$this->isValid($this->TRANSPORTAL_PASSWORD,"TRANSPORTAL_PASSWORD")  && $this->DEBUG != TRUE) {
				$r["STATUS"] = "error";
				$r["DESCRIPTION"] = "The transportal password is invalid. Given is: ".$this->TRANSPORTAL_PASSWORD;
			} elseif(!$this->isValid($this->TERMINAL_RESOURCE_KEY,"TERMINAL_RESOURCE_KEY") && $this->DEBUG != TRUE) {
				$r["STATUS"] = "error";
				$r["DESCRIPTION"] = "The terminal resource key is invalid. Given is: ".$this->TERMINAL_RESOURCE_KEY;
			} elseif(!$this->isValid($this->TERMINAL_URI,"URI") && $this->DEBUG != TRUE) {
				$r["STATUS"] = "error";
				$r["DESCRIPTION"] = "The terminal uri is invalid. Given is: ".$this->TERMINAL_URI;
			} elseif(!$this->isValid($this->CURRENCY,"CURRENCYCODE")) {
				$r["STATUS"] = "error";
				$r["DESCRIPTION"] = "The currency code is invalid. Given is: ".$this->CURRENCY;
			} elseif(!$this->isValid($this->DECIMALS,"DECIMALSCODE")) {
				$r["STATUS"] = "error";
				$r["DESCRIPTION"] = "The unit decimals is invalid. Given is: ".$this->DECIMALS;
			} elseif(!$this->isValid($this->ACTION,"ACTION")) {
				$r["STATUS"] = "error";
				$r["DESCRIPTION"] = "The action code is invalid. Given is: ".$this->ACTION;
			} elseif(!$this->isValid($this->LISTEN_URI,"URI")) {
				$r["STATUS"] = "error";
				$r["DESCRIPTION"] = "The response URI is invalid. Given is: ".$this->LISTEN_URI;
			} elseif(!$this->isValid($this->COMPLETE_URI,"URI")) {
				$r["STATUS"] = "error";
				$r["DESCRIPTION"] = "The completion URI is invalid. Given is: ".$this->COMPLETE_URI;
			} else {
				$this->setResponseToken();
				$r["STATUS"] = "success";
				$r["DESCRIPTION"] = "Access parameters passed validation.";
			}
			return $r;
		}
		
		private function validateInitializationData($p) {
			if(!($this->isValid($p["AMOUNT"],"AMOUNT"))) {
				$p["STATUS"] = "error";
				$p["DESCRIPTION"] = "The processed amount is invalid.";
			} elseif(!($this->isValid($p["LANGUAGE"],"LANGUAGE"))) {
				$p["STATUS"] = "error";
				$p["DESCRIPTION"] = "The processed language id is invalid.";
			} elseif(!($this->isValid($p["USER_DEFINED_VAR_1"],"USER_DEFINED_VAR"))) {
				$p["STATUS"] = "error";
				$p["DESCRIPTION"] = "The processed user defined variable 1 id is invalid. Accepts only letters and numbers, no spaces.";
			} elseif(!($this->isValid($p["USER_DEFINED_VAR_2"],"USER_DEFINED_VAR"))) {
				$p["STATUS"] = "error";
				$p["DESCRIPTION"] = "The processed user defined variable 2 id is invalid. Accepts only letters and numbers, no spaces.";
			} elseif(!($this->isValid($p["USER_DEFINED_VAR_3"],"USER_DEFINED_VAR"))) {
				$p["STATUS"] = "error";
				$p["DESCRIPTION"] = "The processed user defined variable 3 id is invalid. Accepts only letters and numbers, no spaces.";
			} elseif(!($this->isValid($p["USER_DEFINED_VAR_4"],"USER_DEFINED_VAR"))) {
				$p["STATUS"] = "error";
				$p["DESCRIPTION"] = "The processed user defined variable 4 id is invalid. Accepts only letters and numbers, no spaces.";
			} else {
				$p["AMOUNT"] = number_format($p["AMOUNT"],$this->DECIMALS,".","");
				$p["LANGUAGE"] = ($p["LANGUAGE"] == "ARA") ? "ARA" : "ENG";
				$p["STATUS"] = "passed";
				$p["DESCRIPTION"] = "Request parameters passed validation.";
			}
			return $p;
		}
		
		private function setResponseToken() {
			$this->RESPONSETOKEN = md5($this->CURRENCY.":".$this->TRANSPORTAL_ID.":".$this->TRANSPORTAL_PASSWORD.":".$this->TERMINAL_RESOURCE_KEY);
		}

		private function array2Args($a) {
			$s = "";
			foreach ($a as $k => $v) $s .= "&".$k."=".$v;
			return $s;	
		}
		
		private function textEncrypt($s,$k) {
			$pad = 16 - (strlen($s) % 16);
			$s = $s . str_repeat(chr($pad), $pad);			
			return urlencode(bin2hex(join(array_map("chr", unpack('C*',(base64_decode(openssl_encrypt($s,'AES-128-CBC',$k,OPENSSL_ZERO_PADDING,$k))))))));
		}

		private function textDecrypt($code,$key) { 
			$code = $this->hex2ByteArray(trim($code));
			$code = $this->byteArray2String($code);
			$iv = $key; 
			$code = base64_encode($code);
			$decrypted = openssl_decrypt($code, 'AES-128-CBC', $key, OPENSSL_ZERO_PADDING, $iv);
			return $this->pkcs5_unpad($decrypted);
		}
    
		private function hex2ByteArray($hexString) {
			$string = hex2bin($hexString);
			return unpack('C*', $string);
		}

		private function byteArray2String($byteArray) {
			$chars = array_map("chr", $byteArray);
			return join($chars);
		}

		private function pkcs5_unpad($text) {
			$pad = ord($text[strlen($text)-1]);
			if($pad > strlen($text)) {
				return false;	
			}
			if(strspn($text, chr($pad), strlen($text) - $pad) != $pad) {
				return false;
			}
			return substr($text, 0, -1 * $pad);
		}

	}	
?>